#!/usr/bin/env bash

set -e

SCRIPT_BASE_PATH=$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )
cd "$SCRIPT_BASE_PATH"

# URL of the XML file
XML_URL="https://auth.london2038.com/patcher/checksums.xml"

# Base directory to store downloaded files
BASE_DIR="."

# Download the XML file
curl -s "$XML_URL" -o checksums.xml

# Create the download directory if it doesn't exist
mkdir -p "$BASE_DIR"

# Parse the XML manually using grep, sed, and awk
grep '<file ' checksums.xml | sed -n 's/.*name="\([^"]*\)".*hash="\([^"]*\)".*download="\([^"]*\)".*/\1 \2 \3/p' |
while read -r filename hash download; do
    # Only process files marked for download
    if [[ "$download" == "true" ]]; then
        filename=$(echo "$filename" | sed 's|\\|/|g')
        filepath="$BASE_DIR/$filename"
        file_url="https://auth.london2038.com/patcher/${filename}"

        # Create the necessary directories
        mkdir -p "$(dirname "$filepath")"
        
        # Convert expected hash to uppercase
        expected_checksum=$(echo "$hash" | tr '[:lower:]' '[:upper:]')
        
        # Check if file exists
        if [[ -f "$filepath" ]]; then
            # Compute checksum and convert to uppercase
            computed_checksum=$(md5sum "$filepath" | awk '{print toupper($1)}' | tr -d '\\')

            # Verify checksum
            if [[ "$computed_checksum" == "$expected_checksum" ]]; then
                echo "✅ $filename already exists and checksum matches. Skipping download."
                continue
            else
                echo "⚠️ $filename exists but checksum mismatch! Downloading again..."
            fi
        fi

        echo "Downloading $(basename "$filename")..."
        curl -s "$file_url" -o "$filepath"

        # Compute checksum and convert to uppercase
        computed_checksum=$(md5sum "$filepath" | awk '{print toupper($1)}' | tr -d '\\')

        # Verify checksum
        if [[ "$computed_checksum" == "$expected_checksum" ]]; then
            echo "✅ $filename checksum verified!"
        else
            echo "❌ $filename checksum mismatch! Expected: $expected_checksum, Got: $computed_checksum"
        fi
    fi
done

# Cleanup
rm checksums.xml
